<?php

/**
 * JCH Optimize - Performs several front-end optimizations for fast downloads
 *
 * @package   jchoptimize/core
 * @author    Samuel Marshall <samuel@jch-optimize.net>
 * @copyright Copyright (c) 2020 Samuel Marshall / JCH Optimize
 * @license   GNU/GPLv3, or later. See LICENSE file
 *
 * If LICENSE file missing, see <http://www.gnu.org/licenses/>.
 */

namespace JchOptimize\Core\Admin\Ajax;

use JchOptimize\Core\Admin\Helper as AdminHelper;
use JchOptimize\Platform\Paths;
use JchOptimize\Platform\Utility;

defined( '_JEXEC' ) or die( 'Restricted access' );

class FileTree extends Ajax
{

	/**
	 *
	 * @return string
	 */
	public function run()
	{
		//Website document root
		$root = Paths::rootPath();
		//The expanded directory in the folder tree
		$dir = urldecode( Utility::get( 'dir', '', 'string', 'get' ) );
		//Which side of the Explorer view are we rendering? Folder tree or subdirectories and files
		$view = urldecode( Utility::get( 'jchview', '', 'string', 'get' ) );
		//Will be set to 1 if this is the root directory
		$initial = urldecode( Utility::get( 'initial', '0', 'string', 'get' ) );

		//$dir = Utility::decrypt($dir) . '/';
		$dir = $dir . '/';

		if ( $view != 'tree' )
		{//Subdirectories and files
			//Render the header section of the subdirectories side of Explorer
			$header  = '<div id="files-container-header"><ul class="jqueryFileTree"><li><span>&lt;root&gt;' . $dir . '</span></li>';
			$header  .= '<li class="check-all"><span><input type="checkbox"></span><span><em>Check all</em></span>'
			            . '<span><em>' . Utility::translate( 'Width' ) . ' (px)</em></span>'
			            . '<span><em>' . Utility::translate( 'Height' ) . ' (px)</em></span></li></ul></div><div class="files-content">';
			$display = '';
		}
		else
		{
			$display = 'style="display: none;"';
			$header  = '';
		}

		$response = '';

		if ( @file_exists( $root . $dir ) )
		{
			$files = array_diff( scandir( $root . $dir ), array( '..', '.' ) );
//                        $files = Utility::lsFiles($root . $dir, '\.(?:gif|jpe?g|png)$', false);
			natcasesort( $files );

			if ( count( $files ) > 0 )
			{
				$response .= $header;

				if ( $initial && $view == 'tree' )
				{
					$response .= '<div class="files-content"><ul class="jqueryFileTree">';
					$response .= '<li class="directory expanded root">'
					             . '<a href="#" data-root="' . $root . '" data-url="">&lt;root&gt;</a>';
				}

				$response .= '<ul class="jqueryFileTree" ' . $display . '>';

				$directories = array();
				$imagefiles  = array();

				//Initialize counters
				$i = 0;
				$j = 0;

				foreach ( $files as $file )
				{
					if ( is_dir( $root . $dir . $file ) && $file != 'jch_optimize_backup_images' && $file != '.jch' )
					{
						//Limit number of subfolders in Explorer view to 500
						$i ++;

						if ( $i > 500 )
						{
							continue;
						}

						$directories[] = '<li class="directory collapsed">'
						                 . $this->item( $file, $dir, $view, 'dir' ) . '</li>';
					}
					// All files
					elseif ( $view != 'tree' && preg_match( '#\.(?:gif|jpe?g|png)$#i', $file ) && @file_exists( $root . $dir . $file ) )
					{
						//Limit number of images in Explorer view to 1000
						$j ++;

						if ( $j > 1000 )
						{
							continue;
						}

						$ext          = preg_replace( '/^.*\./', '', $file );
						$imagefiles[] = '<li class="file ext_' . strtolower( $ext ) . '">'
						                . $this->item( $file, $dir, $view, 'file' )
						                . '</li>';

					}
				}

				$response .= implode( '', $directories ) . implode( '', $imagefiles ) . '</ul>';

				if ( $initial && $view == 'tree' )
				{
					$response .= '</li></ul></div>';
				}
			}
		}

		return $response;
	}

	/**
	 *
	 * @param   string  $file
	 * @param   string  $dir
	 * @param   string  $view
	 * @param   string  $path
	 *
	 * @return string
	 */
	private function item( $file, $dir, $view, $path )
	{
		$file_path = $dir . $file;
		$root      = Paths::rootPath();

		$anchor = '<a href="#" data-url="' . $file_path . '">'
		          . htmlentities( $file )
		          . '</a>';

		$html = '';

		if ( $view == 'tree' )
		{
			$html .= $anchor;
		}
		else
		{
			if ( $path == 'dir' )
			{
				$html .= '<span><input type="checkbox" value="' . $file_path . '"></span>';
				$html .= $anchor;
			}
			else
			{
				$html .= '<span><input type="checkbox" value="' . $file_path . '"></span>';
				$html .= '<span';

				if ( in_array( $root . $dir . $file, AdminHelper::getOptimizedFiles() ) )
				{
					$html .= ' style="color: blue; font-style: italic;"';
				}

				$html .= '>' . htmlentities( $file ) . '</span>'
				         . '<span><input type="text" size="10" maxlength="5" name="width" ></span>'
				         . '<span><input type="text" size="10" maxlength="5" name="height" ></span>';
			}
		}

		return $html;
	}

}